import org.openqa.selenium.WebDriver;
import org.testng.ITestContext;
import org.testng.ITestListener;
import org.testng.ITestResult;

/**
 * TestNG listener for automatic screenshot capture on test failures.
 * Register this listener in testng.xml or use @Listeners annotation.
 *
 * Usage in testng.xml:
 * <listeners>
 *     <listener class-name="TestNGScreenshotListener"/>
 * </listeners>
 *
 * Usage with annotation:
 * @Listeners(TestNGScreenshotListener.class)
 * public class MyTest { ... }
 */
public class TestNGScreenshotListener implements ITestListener {

    /**
     * Driver attribute name in TestContext
     * Override this if your tests use a different attribute name
     */
    private static final String DRIVER_ATTRIBUTE = "driver";

    @Override
    public void onTestStart(ITestResult result) {
        ScreenshotConfig.log("Test started: " + result.getName());
    }

    @Override
    public void onTestSuccess(ITestResult result) {
        ScreenshotConfig.log("Test passed: " + result.getName());

        // Optional: Capture screenshots for successful critical tests
        if (isCriticalTest(result) && !ScreenshotConfig.CAPTURE_ON_FAILURE_ONLY) {
            captureScreenshot(result, "PASSED");
        }
    }

    @Override
    public void onTestFailure(ITestResult result) {
        ScreenshotConfig.log("Test failed: " + result.getName());

        if (ScreenshotConfig.isEnabled()) {
            captureScreenshot(result, "FAILED");
        }
    }

    @Override
    public void onTestSkipped(ITestResult result) {
        ScreenshotConfig.log("Test skipped: " + result.getName());
    }

    @Override
    public void onTestFailedButWithinSuccessPercentage(ITestResult result) {
        ScreenshotConfig.log("Test failed but within success percentage: " + result.getName());
    }

    @Override
    public void onStart(ITestContext context) {
        ScreenshotConfig.log("Test suite started: " + context.getName());
    }

    @Override
    public void onFinish(ITestContext context) {
        ScreenshotConfig.log("Test suite finished: " + context.getName());
        ScreenshotConfig.log("Tests run: " + context.getAllTestMethods().length);
        ScreenshotConfig.log("Passed: " + context.getPassedTests().size());
        ScreenshotConfig.log("Failed: " + context.getFailedTests().size());
        ScreenshotConfig.log("Skipped: " + context.getSkippedTests().size());
    }

    /**
     * Capture screenshot for a test result
     *
     * @param result Test result
     * @param status Test status (PASSED/FAILED/SKIPPED)
     */
    private void captureScreenshot(ITestResult result, String status) {
        try {
            WebDriver driver = getDriverFromContext(result);

            if (driver != null) {
                String testName = result.getMethod().getMethodName();
                String suiteName = result.getTestContext().getSuite().getName();
                String screenshotName = suiteName + "_" + testName + "_" + status;

                String path = ScreenshotUtil.captureBrowserScreenshot(
                        driver,
                        screenshotName,
                        ScreenshotConfig.getSuiteDirectory(suiteName)
                );

                if (path != null) {
                    ScreenshotConfig.log("Screenshot saved: " + path);
                    // Attach path to test result for reporting
                    result.setAttribute("screenshotPath", path);
                } else {
                    System.err.println("Failed to capture screenshot for: " + testName);
                }
            } else {
                System.err.println("WebDriver not found in test context for: " +
                        result.getMethod().getMethodName());
            }
        } catch (Exception e) {
            System.err.println("Exception while capturing screenshot: " + e.getMessage());
            e.printStackTrace();
        }
    }

    /**
     * Get WebDriver instance from test context
     *
     * @param result Test result
     * @return WebDriver instance, or null if not found
     */
    private WebDriver getDriverFromContext(ITestResult result) {
        // Try to get driver from test context
        Object driver = result.getTestContext().getAttribute(DRIVER_ATTRIBUTE);

        if (driver instanceof WebDriver) {
            return (WebDriver) driver;
        }

        // Try to get driver from test instance (if test class has a driver field)
        try {
            Object testInstance = result.getInstance();
            java.lang.reflect.Field driverField = testInstance.getClass()
                    .getDeclaredField("driver");
            driverField.setAccessible(true);
            Object fieldValue = driverField.get(testInstance);

            if (fieldValue instanceof WebDriver) {
                return (WebDriver) fieldValue;
            }
        } catch (Exception e) {
            // Field not found or not accessible, continue
        }

        return null;
    }

    /**
     * Check if test is marked as critical
     * Override this method to implement custom logic
     *
     * @param result Test result
     * @return true if test is critical, false otherwise
     */
    private boolean isCriticalTest(ITestResult result) {
        // Example: Check if test has @Test(priority = 1) or custom annotation
        return false;
    }
}
