import org.openqa.selenium.*;
import org.openqa.selenium.io.FileHandler;

import java.io.File;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Date;

/**
 * Utility class for capturing screenshots in Selenium WebDriver tests.
 * Supports browser screenshots, element screenshots, and Base64 encoding.
 */
public class ScreenshotUtil {

    private static String getTimestamp() {
        return new SimpleDateFormat(ScreenshotConfig.TIMESTAMP_FORMAT)
                .format(new Date());
    }

    private static File createDirectoryIfNotExists(String directory) {
        File dir = new File(directory);
        if (!dir.exists()) {
            dir.mkdirs();
        }
        return dir;
    }

    /**
     * Capture full browser screenshot with default directory
     *
     * @param driver WebDriver instance
     * @param name   Screenshot name (without extension)
     * @return Path to saved screenshot, or null if failed
     */
    public static String captureBrowserScreenshot(WebDriver driver, String name) {
        return captureBrowserScreenshot(driver, name, ScreenshotConfig.SCREENSHOT_DIR);
    }

    /**
     * Capture full browser screenshot with custom directory
     *
     * @param driver    WebDriver instance
     * @param name      Screenshot name (without extension)
     * @param directory Custom directory path
     * @return Path to saved screenshot, or null if failed
     */
    public static String captureBrowserScreenshot(
            WebDriver driver,
            String name,
            String directory
    ) {
        if (driver == null) {
            System.err.println("WebDriver is null. Screenshot skipped.");
            return null;
        }

        createDirectoryIfNotExists(directory);
        String filename = name + "_" + getTimestamp() + ".png";
        String path = directory + filename;

        try {
            File src = ((TakesScreenshot) driver).getScreenshotAs(OutputType.FILE);
            FileHandler.copy(src, new File(path));
            return path;
        } catch (IOException e) {
            System.err.println("Screenshot failed: " + e.getMessage());
            return null;
        }
    }

    /**
     * Capture screenshot of a specific WebElement
     *
     * @param element WebElement to capture
     * @param name    Screenshot name (without extension)
     * @return Path to saved screenshot, or null if failed
     */
    public static String captureElementScreenshot(WebElement element, String name) {
        return captureElementScreenshot(element, name, ScreenshotConfig.SCREENSHOT_DIR);
    }

    /**
     * Capture screenshot of a specific WebElement with custom directory
     *
     * @param element   WebElement to capture
     * @param name      Screenshot name (without extension)
     * @param directory Custom directory path
     * @return Path to saved screenshot, or null if failed
     */
    public static String captureElementScreenshot(
            WebElement element,
            String name,
            String directory
    ) {
        if (element == null) {
            System.err.println("WebElement is null. Screenshot skipped.");
            return null;
        }

        createDirectoryIfNotExists(directory);
        String filename = name + "_" + getTimestamp() + ".png";
        String path = directory + filename;

        try {
            File src = element.getScreenshotAs(OutputType.FILE);
            FileHandler.copy(src, new File(path));
            return path;
        } catch (IOException e) {
            System.err.println("Element screenshot failed: " + e.getMessage());
            return null;
        }
    }

    /**
     * Capture screenshot as Base64 string (useful for API reports)
     *
     * @param driver WebDriver instance
     * @return Base64 encoded screenshot, or null if failed
     */
    public static String captureBase64Screenshot(WebDriver driver) {
        if (driver == null) {
            System.err.println("WebDriver is null. Base64 screenshot skipped.");
            return null;
        }

        try {
            return ((TakesScreenshot) driver).getScreenshotAs(OutputType.BASE64);
        } catch (Exception e) {
            System.err.println("Base64 screenshot failed: " + e.getMessage());
            return null;
        }
    }

    /**
     * Capture screenshot with retry mechanism
     *
     * @param driver     WebDriver instance
     * @param name       Screenshot name
     * @param maxRetries Maximum retry attempts
     * @return Path to saved screenshot, or null if all retries failed
     */
    public static String captureWithRetry(
            WebDriver driver,
            String name,
            int maxRetries
    ) {
        int attempts = 0;
        Exception lastException = null;

        while (attempts < maxRetries) {
            try {
                return captureBrowserScreenshot(driver, name);
            } catch (Exception e) {
                lastException = e;
                attempts++;
                try {
                    Thread.sleep(1000 * attempts);
                } catch (InterruptedException ie) {
                    Thread.currentThread().interrupt();
                }
            }
        }

        System.err.println("Screenshot failed after " + maxRetries +
                " retries: " + (lastException != null ? lastException.getMessage() : "Unknown error"));
        return null;
    }

    /**
     * Capture screenshot for parallel execution (thread-safe)
     *
     * @param driver WebDriver instance
     * @param name   Screenshot name
     * @return Path to saved screenshot
     */
    public static synchronized String captureThreadSafeScreenshot(
            WebDriver driver,
            String name
    ) {
        String threadId = Thread.currentThread().getId() + "_";
        return captureBrowserScreenshot(driver, threadId + name);
    }
}
