/**
 * Configuration class for ScreenshotUtil.
 * Centralized settings for screenshot directory, naming, and behavior.
 */
public class ScreenshotConfig {

    /**
     * Default directory for storing screenshots
     * Change this to customize storage location
     */
    public static final String SCREENSHOT_DIR = "screenshots/";

    /**
     * Timestamp format for screenshot filenames
     * Format: yyyyMMdd_HHmmss
     * Example: 20251224_143052
     */
    public static final String TIMESTAMP_FORMAT = "yyyyMMdd_HHmmss";

    /**
     * Screenshot file extension
     */
    public static final String FILE_EXTENSION = ".png";

    /**
     * Enable/disable screenshot capture
     * Set to false to disable all screenshots (useful for debugging)
     */
    public static boolean ENABLED = true;

    /**
     * Capture screenshots only on test failures
     * Set to true to reduce screenshot noise in CI/CD
     */
    public static boolean CAPTURE_ON_FAILURE_ONLY = true;

    /**
     * Maximum retry attempts for screenshot capture
     */
    public static final int MAX_RETRIES = 3;

    /**
     * Enable verbose logging for screenshot operations
     */
    public static boolean VERBOSE_LOGGING = false;

    // Private constructor to prevent instantiation
    private ScreenshotConfig() {
        throw new UnsupportedOperationException("Configuration class cannot be instantiated");
    }

    /**
     * Get screenshot directory for a specific test suite
     *
     * @param suiteName Test suite name
     * @return Directory path for the suite
     */
    public static String getSuiteDirectory(String suiteName) {
        return SCREENSHOT_DIR + suiteName + "/";
    }

    /**
     * Check if screenshot capture is enabled
     *
     * @return true if enabled, false otherwise
     */
    public static boolean isEnabled() {
        return ENABLED;
    }

    /**
     * Enable or disable screenshot capture globally
     *
     * @param enabled true to enable, false to disable
     */
    public static void setEnabled(boolean enabled) {
        ENABLED = enabled;
        if (VERBOSE_LOGGING) {
            System.out.println("Screenshot capture " +
                    (enabled ? "enabled" : "disabled"));
        }
    }

    /**
     * Log message if verbose logging is enabled
     *
     * @param message Message to log
     */
    public static void log(String message) {
        if (VERBOSE_LOGGING) {
            System.out.println("[ScreenshotUtil] " + message);
        }
    }
}
