# Java Excel Utility - Complete Guide

A comprehensive Excel utility package for Java automation testing with Apache POI. Supports both `.xlsx` and `.xls` formats with read, write, and styling operations.

## 📦 Package Contents

- **ExcelUtil.java** - Base utility class with core read/write operations
- **ExcelReader.java** - Extended reading operations (2D arrays, maps, search)
- **ExcelWriter.java** - Extended writing operations with styling support
- **ExcelConstants.java** - Constants for styles, colors, and common values
- **ExcelUtilDemo.java** - Comprehensive usage examples
- **pom.xml** - Maven dependencies configuration
- **README.md** - This documentation file

## 🚀 Quick Start

### 1. Setup

**Option A: Maven Project**

Add the dependencies from `pom.xml` to your project:

```xml
<dependency>
    <groupId>org.apache.poi</groupId>
    <artifactId>poi</artifactId>
    <version>5.2.5</version>
</dependency>
<dependency>
    <groupId>org.apache.poi</groupId>
    <artifactId>poi-ooxml</artifactId>
    <version>5.2.5</version>
</dependency>
```

**Option B: Manual Setup**

1. Copy all `.java` files to your project's `src/main/java/com/automation/utils/` directory
2. Download Apache POI JARs from [Apache POI Downloads](https://poi.apache.org/download.html)
3. Add JARs to your project's build path

### 2. Basic Usage

```java
// Reading data
ExcelUtil excel = new ExcelUtil("path/to/file.xlsx");
String value = excel.getCellData("Sheet1", 0, 0);
excel.close();

// Writing data
ExcelWriter writer = new ExcelWriter("output.xlsx", true);
writer.writeRow("Sheet1", 0, new String[]{"Name", "Age", "Email"});
writer.close();
```

## 📚 API Documentation

### ExcelUtil (Base Class)

Core functionality for basic Excel operations.

#### Constructor
```java
ExcelUtil(String filePath)
```
Opens an existing Excel file (.xlsx or .xls).

#### Methods

| Method | Description | Returns |
|--------|-------------|---------|
| `getRowCount(String sheetName)` | Get total rows in sheet | int |
| `getColumnCount(String sheetName)` | Get total columns in sheet | int |
| `getCellData(String sheetName, int row, int col)` | Read cell value as String | String |
| `setCellData(String sheetName, int row, int col, String data)` | Write value to cell | boolean |
| `isSheetExist(String sheetName)` | Check if sheet exists | boolean |
| `getAllSheetNames()` | Get list of all sheet names | List&lt;String&gt; |
| `findRowByColumnValue(String sheet, int col, String value)` | Find row number by value | int |
| `close()` | Close workbook and streams | void |

**Example:**
```java
ExcelUtil excel = new ExcelUtil("TestData.xlsx");

// Get sheet info
int rows = excel.getRowCount("Users");
int cols = excel.getColumnCount("Users");

// Read cell
String username = excel.getCellData("Users", 1, 0);

// Write cell
excel.setCellData("Users", 1, 3, "Updated");

// Find row
int row = excel.findRowByColumnValue("Users", 0, "john.doe");

excel.close();
```

---

### ExcelReader (Extended Reading)

Advanced reading operations for complex data structures.

#### Constructor
```java
ExcelReader(String filePath)
```
Extends ExcelUtil with additional reading methods.

#### Methods

| Method | Description | Returns |
|--------|-------------|---------|
| `readSheet(String sheetName)` | Read entire sheet as 2D array | String[][] |
| `readSheetAsMap(String sheetName)` | Read sheet as List of Maps (headers as keys) | List&lt;Map&lt;String,String&gt;&gt; |
| `getTestData(String sheet, String testCaseName)` | Get test data row by test case name | Map&lt;String,String&gt; |
| `readRow(String sheetName, int rowNum)` | Read specific row as array | String[] |
| `readColumn(String sheetName, int colNum)` | Read specific column as array | String[] |
| `getColumnIndex(String sheet, String headerName)` | Get column index by header name | int |
| `getCellDataByHeader(String sheet, int row, String header)` | Read cell using header name | String |
| `findAllRowsByColumnValue(String sheet, int col, String value)` | Find all rows with matching value | List&lt;Integer&gt; |
| `isCellEmpty(String sheet, int row, int col)` | Check if cell is empty | boolean |
| `readSheetFromRow(String sheet, int startRow)` | Read data starting from specific row | String[][] |

**Example:**
```java
ExcelReader reader = new ExcelReader("TestData.xlsx");

// Read entire sheet as 2D array
String[][] data = reader.readSheet("TestCases");

// Read as List of Maps (useful for iteration)
List<Map<String, String>> testCases = reader.readSheetAsMap("TestCases");
for (Map<String, String> testCase : testCases) {
    String id = testCase.get("TestCaseID");
    String name = testCase.get("TestCaseName");
}

// Get specific test data
Map<String, String> tc001 = reader.getTestData("TestCases", "TC001");
String username = tc001.get("Username");

// Read by header name
String email = reader.getCellDataByHeader("Users", 1, "Email");

reader.close();
```

---

### ExcelWriter (Extended Writing)

Advanced writing operations with styling support.

#### Constructors
```java
ExcelWriter(String filePath)                    // Open existing file
ExcelWriter(String filePath, boolean createNew) // Create new file
```

#### Methods

| Method | Description | Returns |
|--------|-------------|---------|
| `writeRow(String sheet, int row, String[] data)` | Write entire row | boolean |
| `appendRow(String sheet, String[] data)` | Append row at end | boolean |
| `writeSheetData(String sheet, String[][] data, int startRow)` | Write 2D array | boolean |
| `createSheet(String sheetName)` | Create new sheet | boolean |
| `removeSheet(String sheetName)` | Remove sheet | boolean |
| `setCellDataWithStyle(String sheet, int row, int col, String data, CellStyle style)` | Write with style | boolean |
| `createHeaderStyle()` | Create header cell style | CellStyle |
| `createPassStyle()` | Create PASS style (green) | CellStyle |
| `createFailStyle()` | Create FAIL style (red) | CellStyle |
| `createCustomStyle(short bg, short font, boolean bold, short size)` | Create custom style | CellStyle |
| `writeTestResult(String sheet, int row, int col, String result)` | Write PASS/FAIL with style | boolean |
| `autoSizeColumns(String sheetName)` | Auto-size all columns | void |
| `setColumnWidth(String sheet, int col, int width)` | Set column width | void |
| `writeHeaders(String sheet, String[] headers)` | Write headers with style | boolean |
| `clearSheet(String sheetName)` | Clear all data from sheet | boolean |

**Example:**
```java
ExcelWriter writer = new ExcelWriter("Results.xlsx", true);

// Create sheet and write headers
writer.createSheet("TestResults");
String[] headers = {"TestCaseID", "Status", "Remarks"};
writer.writeHeaders("TestResults", headers);

// Write data
String[] row1 = {"TC001", "PASS", "Success"};
writer.writeRow("TestResults", 1, row1);

// Append more rows
writer.appendRow("TestResults", new String[]{"TC002", "FAIL", "Error"});

// Write with styling
writer.writeTestResult("TestResults", 1, 1, "PASS");  // Green background
writer.writeTestResult("TestResults", 2, 1, "FAIL");  // Red background

// Format columns
writer.autoSizeColumns("TestResults");
writer.setColumnWidth("TestResults", 2, 40);  // Remarks column wider

writer.close();
```

---

### ExcelConstants

Pre-defined constants for common values, colors, and configurations.

#### Sheet Names
```java
ExcelConstants.SHEET_TEST_DATA        // "TestData"
ExcelConstants.SHEET_TEST_RESULTS     // "TestResults"
ExcelConstants.SHEET_CONFIG           // "Config"
```

#### Test Results
```java
ExcelConstants.RESULT_PASS            // "PASS"
ExcelConstants.RESULT_FAIL            // "FAIL"
ExcelConstants.RESULT_SKIP            // "SKIP"
```

#### Column Headers
```java
ExcelConstants.COL_TEST_CASE_ID       // "TestCaseID"
ExcelConstants.COL_STATUS             // "Status"
ExcelConstants.COL_USERNAME           // "Username"
```

#### Colors
```java
ExcelConstants.COLOR_PASS_BG          // Green background
ExcelConstants.COLOR_FAIL_BG          // Red background
ExcelConstants.COLOR_HEADER_BG        // Dark blue background
```

---

## 🎯 Common Use Cases

### 1. Reading Test Data for TestNG

```java
@DataProvider(name = "testData")
public Object[][] getTestData() {
    ExcelReader reader = new ExcelReader("src/test/resources/TestData.xlsx");
    String[][] data = reader.readSheetFromRow("LoginTests", 1); // Skip header
    reader.close();

    Object[][] testData = new Object[data.length][];
    for (int i = 0; i < data.length; i++) {
        testData[i] = data[i];
    }
    return testData;
}

@Test(dataProvider = "testData")
public void loginTest(String username, String password, String expectedResult) {
    // Your test logic here
}
```

### 2. Writing Test Execution Results

```java
public void writeTestResult(String testCaseId, boolean passed) {
    ExcelWriter writer = new ExcelWriter("test-output/Results.xlsx");

    int row = writer.findRowByColumnValue("Results", 0, testCaseId);
    if (row != -1) {
        String status = passed ? "PASS" : "FAIL";
        writer.writeTestResult("Results", row, 2, status);
        writer.setCellData("Results", row, 3, new Date().toString());
    }

    writer.close();
}
```

### 3. Data-Driven Testing

```java
ExcelReader reader = new ExcelReader("TestData.xlsx");
List<Map<String, String>> testCases = reader.readSheetAsMap("TestCases");

for (Map<String, String> testCase : testCases) {
    if ("Yes".equals(testCase.get("RunMode"))) {
        String url = testCase.get("URL");
        String username = testCase.get("Username");
        String password = testCase.get("Password");

        // Execute test with this data
        executeTest(url, username, password);
    }
}
reader.close();
```

### 4. Creating Test Report

```java
ExcelWriter report = new ExcelWriter("TestReport.xlsx", true);
report.createSheet("Summary");

// Write headers
String[] headers = {"Test Suite", "Total", "Pass", "Fail", "Pass %"};
report.writeHeaders("Summary", headers);

// Write summary data
String[] suite1 = {"Login Tests", "10", "8", "2", "80%"};
report.writeRow("Summary", 1, suite1);

// Apply styling
report.writeTestResult("Summary", 1, 2, "8");  // Pass count in green
report.writeTestResult("Summary", 1, 3, "2");  // Fail count in red

report.autoSizeColumns("Summary");
report.close();
```

### 5. Comparing Expected vs Actual Results

```java
ExcelReader reader = new ExcelReader("TestData.xlsx");
ExcelWriter writer = new ExcelWriter("TestData.xlsx");

int rowCount = reader.getRowCount("Results");

for (int i = 1; i < rowCount; i++) {
    String expected = reader.getCellData("Results", i, 3);
    String actual = reader.getCellData("Results", i, 4);

    if (expected.equals(actual)) {
        writer.writeTestResult("Results", i, 5, "PASS");
    } else {
        writer.writeTestResult("Results", i, 5, "FAIL");
    }
}

reader.close();
writer.close();
```

---

## 🔧 Advanced Examples

### Custom Styling

```java
ExcelWriter writer = new ExcelWriter("Styled.xlsx", true);

// Create custom style
CellStyle customStyle = writer.createCustomStyle(
    IndexedColors.YELLOW.getIndex(),  // Background
    IndexedColors.BLACK.getIndex(),   // Font color
    true,                              // Bold
    (short) 14                         // Font size
);

// Apply custom style
writer.setCellDataWithStyle("Sheet1", 0, 0, "Important!", customStyle);

writer.close();
```

### Batch Operations

```java
ExcelWriter writer = new ExcelWriter("Batch.xlsx", true);
writer.createSheet("Data");

// Write bulk data efficiently
String[][] bulkData = new String[1000][5];
for (int i = 0; i < 1000; i++) {
    bulkData[i] = new String[]{
        "ID" + i, "Name" + i, "Value" + i, "Status", "Remarks"
    };
}

writer.writeSheetData("Data", bulkData, 1);
writer.autoSizeColumns("Data");

writer.close();
```

### Multi-Sheet Operations

```java
ExcelWriter writer = new ExcelWriter("MultiSheet.xlsx", true);

// Create multiple sheets
for (String suite : new String[]{"Smoke", "Regression", "Sanity"}) {
    writer.createSheet(suite);
    writer.writeHeaders(suite, new String[]{"TestCase", "Status"});
}

// Get all sheets
List<String> sheets = writer.getAllSheetNames();
System.out.println("Created sheets: " + sheets);

writer.close();
```

---

## 📋 Best Practices

1. **Always close resources**
   ```java
   ExcelUtil excel = new ExcelUtil("file.xlsx");
   try {
       // Your operations
   } finally {
       excel.close();
   }
   ```

2. **Use ExcelReader for reading operations**
   - More efficient methods for data retrieval
   - Better handling of different data structures

3. **Use ExcelWriter for writing operations**
   - Automatic styling support
   - Batch write operations

4. **Leverage ExcelConstants**
   - Maintain consistency across your test suite
   - Easy to update common values

5. **Handle exceptions properly**
   - Methods return `boolean` for success/failure
   - Check return values before proceeding

6. **Optimize performance**
   - Use `readSheetAsMap()` for iteration
   - Use `writeSheetData()` for bulk writes
   - Call `autoSizeColumns()` only once after all writes

---

## 🐛 Troubleshooting

### FileNotFoundException
```java
// Ensure file path is correct (use absolute path if needed)
String filePath = new File("TestData.xlsx").getAbsolutePath();
ExcelUtil excel = new ExcelUtil(filePath);
```

### Sheet not found
```java
// Check if sheet exists before using
if (excel.isSheetExist("MySheet")) {
    String value = excel.getCellData("MySheet", 0, 0);
}
```

### Memory issues with large files
```java
// For very large files, read in chunks
ExcelReader reader = new ExcelReader("large.xlsx");
for (int i = 0; i < rowCount; i += 100) {
    String[][] chunk = reader.readSheetFromRow("Data", i);
    // Process chunk
}
```

### Cell format issues
```java
// getCellValue() handles all formats automatically
// Returns String representation of any cell type
```

---

## 📄 License

This utility is provided as-is for educational and commercial use.

## 🤝 Contributing

Feel free to extend these utilities for your specific needs:
- Add new styling methods
- Implement formula support
- Add data validation
- Create report templates

---

## 📞 Support

For issues, questions, or contributions, refer to the source documentation or create an issue in your project repository.

---

**Happy Testing! 🚀**
