package com.automation.utils;

import org.apache.poi.ss.usermodel.*;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;

import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.Map;

/**
 * Advanced Excel Writing Operations with Styling
 * Extends ExcelUtil with methods for writing data and applying styles
 *
 * @author DevTools
 * @version 1.0
 */
public class ExcelWriter extends ExcelUtil {

    /**
     * Constructor - Opens existing Excel file
     * @param filePath Path to Excel file (.xlsx or .xls)
     */
    public ExcelWriter(String filePath) {
        super(filePath);
    }

    /**
     * Constructor - Creates new Excel file
     * @param filePath Path where new file will be created
     * @param createNew If true, creates new file even if it exists
     */
    public ExcelWriter(String filePath, boolean createNew) {
        this.filePath = filePath;
        try {
            File file = new File(filePath);

            if (createNew || !file.exists()) {
                if (filePath.endsWith(".xlsx")) {
                    workbook = new XSSFWorkbook();
                } else {
                    throw new IllegalArgumentException("Use .xlsx format for new files");
                }
            } else {
                // Open existing file
                super.filePath = filePath;
                fis = new java.io.FileInputStream(file);
                if (filePath.endsWith(".xlsx")) {
                    workbook = new XSSFWorkbook(fis);
                } else {
                    workbook = new org.apache.poi.hssf.usermodel.HSSFWorkbook(fis);
                }
            }
        } catch (Exception e) {
            System.err.println("Error creating/opening Excel file: " + e.getMessage());
            e.printStackTrace();
        }
    }

    /**
     * Write entire row of data
     * @param sheetName Name of the sheet
     * @param rowNum Row number (0-based)
     * @param data Array of values to write
     * @return true if successful
     */
    public boolean writeRow(String sheetName, int rowNum, String[] data) {
        try {
            sheet = workbook.getSheet(sheetName);
            if (sheet == null) {
                sheet = workbook.createSheet(sheetName);
            }

            Row row = sheet.getRow(rowNum);
            if (row == null) {
                row = sheet.createRow(rowNum);
            }

            for (int i = 0; i < data.length; i++) {
                Cell cell = row.getCell(i);
                if (cell == null) {
                    cell = row.createCell(i);
                }
                cell.setCellValue(data[i]);
            }

            saveWorkbook();
            return true;
        } catch (Exception e) {
            System.err.println("Error writing row: " + e.getMessage());
            e.printStackTrace();
            return false;
        }
    }

    /**
     * Append row at the end of sheet
     * @param sheetName Name of the sheet
     * @param data Array of values to append
     * @return true if successful
     */
    public boolean appendRow(String sheetName, String[] data) {
        int rowCount = getRowCount(sheetName);
        return writeRow(sheetName, rowCount, data);
    }

    /**
     * Write 2D array data to sheet
     * @param sheetName Name of the sheet
     * @param data 2D array of data
     * @param startRow Starting row number (0-based)
     * @return true if successful
     */
    public boolean writeSheetData(String sheetName, String[][] data, int startRow) {
        try {
            for (int i = 0; i < data.length; i++) {
                if (!writeRow(sheetName, startRow + i, data[i])) {
                    return false;
                }
            }
            return true;
        } catch (Exception e) {
            System.err.println("Error writing sheet data: " + e.getMessage());
            return false;
        }
    }

    /**
     * Create new sheet in workbook
     * @param sheetName Name for the new sheet
     * @return true if successful
     */
    public boolean createSheet(String sheetName) {
        try {
            if (isSheetExist(sheetName)) {
                System.err.println("Sheet already exists: " + sheetName);
                return false;
            }
            workbook.createSheet(sheetName);
            saveWorkbook();
            return true;
        } catch (Exception e) {
            System.err.println("Error creating sheet: " + e.getMessage());
            return false;
        }
    }

    /**
     * Remove sheet from workbook
     * @param sheetName Name of sheet to remove
     * @return true if successful
     */
    public boolean removeSheet(String sheetName) {
        try {
            if (!isSheetExist(sheetName)) {
                System.err.println("Sheet does not exist: " + sheetName);
                return false;
            }
            int index = workbook.getSheetIndex(sheetName);
            workbook.removeSheetAt(index);
            saveWorkbook();
            return true;
        } catch (Exception e) {
            System.err.println("Error removing sheet: " + e.getMessage());
            return false;
        }
    }

    /**
     * Set cell data with custom style
     * @param sheetName Name of the sheet
     * @param rowNum Row number (0-based)
     * @param colNum Column number (0-based)
     * @param data Data to write
     * @param style CellStyle to apply
     * @return true if successful
     */
    public boolean setCellDataWithStyle(String sheetName, int rowNum, int colNum, String data, CellStyle style) {
        try {
            sheet = workbook.getSheet(sheetName);
            if (sheet == null) {
                sheet = workbook.createSheet(sheetName);
            }

            Row row = sheet.getRow(rowNum);
            if (row == null) {
                row = sheet.createRow(rowNum);
            }

            Cell cell = row.getCell(colNum);
            if (cell == null) {
                cell = row.createCell(colNum);
            }

            cell.setCellValue(data);
            if (style != null) {
                cell.setCellStyle(style);
            }

            saveWorkbook();
            return true;
        } catch (Exception e) {
            System.err.println("Error setting cell data with style: " + e.getMessage());
            return false;
        }
    }

    /**
     * Create header cell style (bold, centered, colored background)
     * @return CellStyle for headers
     */
    public CellStyle createHeaderStyle() {
        try {
            CellStyle style = workbook.createCellStyle();

            // Font settings
            Font font = workbook.createFont();
            font.setBold(true);
            font.setFontHeightInPoints((short) 12);
            font.setColor(IndexedColors.WHITE.getIndex());
            style.setFont(font);

            // Background color
            style.setFillForegroundColor(IndexedColors.DARK_BLUE.getIndex());
            style.setFillPattern(FillPatternType.SOLID_FOREGROUND);

            // Alignment
            style.setAlignment(HorizontalAlignment.CENTER);
            style.setVerticalAlignment(VerticalAlignment.CENTER);

            // Borders
            style.setBorderTop(BorderStyle.THIN);
            style.setBorderBottom(BorderStyle.THIN);
            style.setBorderLeft(BorderStyle.THIN);
            style.setBorderRight(BorderStyle.THIN);

            return style;
        } catch (Exception e) {
            System.err.println("Error creating header style: " + e.getMessage());
            return null;
        }
    }

    /**
     * Create PASS style (green background)
     * @return CellStyle for PASS results
     */
    public CellStyle createPassStyle() {
        try {
            CellStyle style = workbook.createCellStyle();

            Font font = workbook.createFont();
            font.setBold(true);
            font.setColor(IndexedColors.WHITE.getIndex());
            style.setFont(font);

            style.setFillForegroundColor(IndexedColors.GREEN.getIndex());
            style.setFillPattern(FillPatternType.SOLID_FOREGROUND);

            style.setAlignment(HorizontalAlignment.CENTER);

            return style;
        } catch (Exception e) {
            System.err.println("Error creating pass style: " + e.getMessage());
            return null;
        }
    }

    /**
     * Create FAIL style (red background)
     * @return CellStyle for FAIL results
     */
    public CellStyle createFailStyle() {
        try {
            CellStyle style = workbook.createCellStyle();

            Font font = workbook.createFont();
            font.setBold(true);
            font.setColor(IndexedColors.WHITE.getIndex());
            style.setFont(font);

            style.setFillForegroundColor(IndexedColors.RED.getIndex());
            style.setFillPattern(FillPatternType.SOLID_FOREGROUND);

            style.setAlignment(HorizontalAlignment.CENTER);

            return style;
        } catch (Exception e) {
            System.err.println("Error creating fail style: " + e.getMessage());
            return null;
        }
    }

    /**
     * Create custom cell style with specified colors and formatting
     * @param bgColor Background color index
     * @param fontColor Font color index
     * @param bold Whether font should be bold
     * @param fontSize Font size in points
     * @return Custom CellStyle
     */
    public CellStyle createCustomStyle(short bgColor, short fontColor, boolean bold, short fontSize) {
        try {
            CellStyle style = workbook.createCellStyle();

            Font font = workbook.createFont();
            font.setBold(bold);
            font.setFontHeightInPoints(fontSize);
            font.setColor(fontColor);
            style.setFont(font);

            if (bgColor > 0) {
                style.setFillForegroundColor(bgColor);
                style.setFillPattern(FillPatternType.SOLID_FOREGROUND);
            }

            return style;
        } catch (Exception e) {
            System.err.println("Error creating custom style: " + e.getMessage());
            return null;
        }
    }

    /**
     * Write test result with appropriate styling
     * @param sheetName Name of the sheet
     * @param rowNum Row number (0-based)
     * @param colNum Column number (0-based)
     * @param result "PASS" or "FAIL"
     * @return true if successful
     */
    public boolean writeTestResult(String sheetName, int rowNum, int colNum, String result) {
        CellStyle style = result.equalsIgnoreCase("PASS") ? createPassStyle() : createFailStyle();
        return setCellDataWithStyle(sheetName, rowNum, colNum, result, style);
    }

    /**
     * Auto-size all columns in sheet
     * @param sheetName Name of the sheet
     */
    public void autoSizeColumns(String sheetName) {
        try {
            sheet = workbook.getSheet(sheetName);
            if (sheet == null) {
                System.err.println("Sheet not found: " + sheetName);
                return;
            }

            int colCount = getColumnCount(sheetName);
            for (int i = 0; i < colCount; i++) {
                sheet.autoSizeColumn(i);
            }

            saveWorkbook();
        } catch (Exception e) {
            System.err.println("Error auto-sizing columns: " + e.getMessage());
        }
    }

    /**
     * Set column width
     * @param sheetName Name of the sheet
     * @param colNum Column number (0-based)
     * @param width Width in characters (will be converted to Excel units)
     */
    public void setColumnWidth(String sheetName, int colNum, int width) {
        try {
            sheet = workbook.getSheet(sheetName);
            if (sheet == null) {
                System.err.println("Sheet not found: " + sheetName);
                return;
            }

            sheet.setColumnWidth(colNum, width * 256); // Excel uses units of 1/256th of a character width
            saveWorkbook();
        } catch (Exception e) {
            System.err.println("Error setting column width: " + e.getMessage());
        }
    }

    /**
     * Write headers with header style
     * @param sheetName Name of the sheet
     * @param headers Array of header names
     * @return true if successful
     */
    public boolean writeHeaders(String sheetName, String[] headers) {
        try {
            CellStyle headerStyle = createHeaderStyle();
            sheet = workbook.getSheet(sheetName);
            if (sheet == null) {
                sheet = workbook.createSheet(sheetName);
            }

            Row row = sheet.getRow(0);
            if (row == null) {
                row = sheet.createRow(0);
            }

            for (int i = 0; i < headers.length; i++) {
                Cell cell = row.getCell(i);
                if (cell == null) {
                    cell = row.createCell(i);
                }
                cell.setCellValue(headers[i]);
                cell.setCellStyle(headerStyle);
            }

            saveWorkbook();
            return true;
        } catch (Exception e) {
            System.err.println("Error writing headers: " + e.getMessage());
            return false;
        }
    }

    /**
     * Clear all data from sheet (keeps sheet structure)
     * @param sheetName Name of the sheet
     * @return true if successful
     */
    public boolean clearSheet(String sheetName) {
        try {
            sheet = workbook.getSheet(sheetName);
            if (sheet == null) {
                System.err.println("Sheet not found: " + sheetName);
                return false;
            }

            int rowCount = getRowCount(sheetName);
            for (int i = rowCount - 1; i >= 0; i--) {
                Row row = sheet.getRow(i);
                if (row != null) {
                    sheet.removeRow(row);
                }
            }

            saveWorkbook();
            return true;
        } catch (Exception e) {
            System.err.println("Error clearing sheet: " + e.getMessage());
            return false;
        }
    }
}
