package com.automation.utils;

import org.apache.poi.ss.usermodel.CellStyle;

import java.util.List;
import java.util.Map;

/**
 * Demo class showing usage examples of Excel Utilities
 * Demonstrates reading, writing, and styling operations
 *
 * @author DevTools
 * @version 1.0
 */
public class ExcelUtilDemo {

    public static void main(String[] args) {
        // Example file paths
        String testDataFile = "src/test/resources/TestData.xlsx";
        String testResultsFile = "test-output/TestResults.xlsx";

        System.out.println("=== Excel Utility Demo ===\n");

        // Demo 1: Basic Reading Operations
        demoBasicReading(testDataFile);

        // Demo 2: Advanced Reading Operations
        demoAdvancedReading(testDataFile);

        // Demo 3: Writing Operations
        demoWriting(testResultsFile);

        // Demo 4: Styling and Formatting
        demoStyling(testResultsFile);

        // Demo 5: Test Data Integration
        demoTestDataUsage(testDataFile);
    }

    /**
     * Demo 1: Basic Reading Operations
     */
    public static void demoBasicReading(String filePath) {
        System.out.println("--- Demo 1: Basic Reading ---");

        ExcelUtil excel = new ExcelUtil(filePath);

        // Get sheet information
        String sheetName = "TestData";
        int rowCount = excel.getRowCount(sheetName);
        int colCount = excel.getColumnCount(sheetName);

        System.out.println("Sheet: " + sheetName);
        System.out.println("Rows: " + rowCount + ", Columns: " + colCount);

        // Read specific cell
        String value = excel.getCellData(sheetName, 1, 0);
        System.out.println("Cell [1,0]: " + value);

        // Check if sheet exists
        boolean exists = excel.isSheetExist("Config");
        System.out.println("Config sheet exists: " + exists);

        // Get all sheet names
        List<String> sheets = excel.getAllSheetNames();
        System.out.println("All sheets: " + sheets);

        excel.close();
        System.out.println();
    }

    /**
     * Demo 2: Advanced Reading Operations
     */
    public static void demoAdvancedReading(String filePath) {
        System.out.println("--- Demo 2: Advanced Reading ---");

        ExcelReader reader = new ExcelReader(filePath);
        String sheetName = "TestData";

        // Read entire sheet as 2D array
        String[][] allData = reader.readSheet(sheetName);
        System.out.println("Read " + allData.length + " rows of data");

        // Read sheet as List of Maps (with headers)
        List<Map<String, String>> dataList = reader.readSheetAsMap(sheetName);
        System.out.println("Data as Maps: " + dataList.size() + " records");

        if (!dataList.isEmpty()) {
            Map<String, String> firstRow = dataList.get(0);
            System.out.println("First record: " + firstRow);
        }

        // Get specific test data by test case name
        Map<String, String> testData = reader.getTestData(sheetName, "TC001");
        System.out.println("Test Case TC001 data: " + testData);

        // Read specific row
        String[] row = reader.readRow(sheetName, 1);
        System.out.println("Row 1: " + String.join(", ", row));

        // Read specific column
        String[] column = reader.readColumn(sheetName, 0);
        System.out.println("Column 0 has " + column.length + " values");

        // Get column index by header name
        int colIndex = reader.getColumnIndex(sheetName, "TestCaseName");
        System.out.println("TestCaseName column index: " + colIndex);

        // Get cell data by header name
        String cellValue = reader.getCellDataByHeader(sheetName, 1, "Username");
        System.out.println("Username at row 1: " + cellValue);

        // Find rows by value
        List<Integer> rows = reader.findAllRowsByColumnValue(sheetName, 0, "TC001");
        System.out.println("Rows with TC001: " + rows);

        reader.close();
        System.out.println();
    }

    /**
     * Demo 3: Writing Operations
     */
    public static void demoWriting(String filePath) {
        System.out.println("--- Demo 3: Writing Operations ---");

        // Create new Excel file
        ExcelWriter writer = new ExcelWriter(filePath, true);

        // Create new sheet
        writer.createSheet("TestResults");

        // Write headers
        String[] headers = {"TestCaseID", "TestCaseName", "Status", "Remarks"};
        writer.writeHeaders("TestResults", headers);
        System.out.println("Headers written");

        // Write single row
        String[] row1 = {"TC001", "Login Test", "PASS", "Successful login"};
        writer.writeRow("TestResults", 1, row1);
        System.out.println("Row 1 written");

        // Append row
        String[] row2 = {"TC002", "Logout Test", "PASS", "Successful logout"};
        writer.appendRow("TestResults", row2);
        System.out.println("Row 2 appended");

        // Write bulk data
        String[][] bulkData = {
            {"TC003", "Search Test", "FAIL", "No results found"},
            {"TC004", "Filter Test", "PASS", "Filters applied correctly"}
        };
        writer.writeSheetData("TestResults", bulkData, 3);
        System.out.println("Bulk data written");

        // Set specific cell
        writer.setCellData("TestResults", 5, 0, "TC005");
        writer.setCellData("TestResults", 5, 1, "Navigation Test");

        // Auto-size columns
        writer.autoSizeColumns("TestResults");
        System.out.println("Columns auto-sized");

        writer.close();
        System.out.println("File saved: " + filePath);
        System.out.println();
    }

    /**
     * Demo 4: Styling and Formatting
     */
    public static void demoStyling(String filePath) {
        System.out.println("--- Demo 4: Styling and Formatting ---");

        ExcelWriter writer = new ExcelWriter(filePath);
        String sheetName = "TestResults";

        // Create styles
        CellStyle headerStyle = writer.createHeaderStyle();
        CellStyle passStyle = writer.createPassStyle();
        CellStyle failStyle = writer.createFailStyle();

        System.out.println("Styles created");

        // Apply header style to first row
        writer.setCellDataWithStyle(sheetName, 0, 0, "TestCaseID", headerStyle);
        writer.setCellDataWithStyle(sheetName, 0, 1, "TestCaseName", headerStyle);
        writer.setCellDataWithStyle(sheetName, 0, 2, "Status", headerStyle);
        writer.setCellDataWithStyle(sheetName, 0, 3, "Remarks", headerStyle);

        // Write test results with styling
        writer.writeTestResult(sheetName, 1, 2, "PASS");
        writer.writeTestResult(sheetName, 3, 2, "FAIL");

        System.out.println("Styled data written");

        // Set custom column widths
        writer.setColumnWidth(sheetName, 0, 15);  // TestCaseID
        writer.setColumnWidth(sheetName, 1, 30);  // TestCaseName
        writer.setColumnWidth(sheetName, 2, 12);  // Status
        writer.setColumnWidth(sheetName, 3, 40);  // Remarks

        System.out.println("Column widths set");

        writer.close();
        System.out.println();
    }

    /**
     * Demo 5: Test Data Integration Example
     */
    public static void demoTestDataUsage(String filePath) {
        System.out.println("--- Demo 5: Test Data Integration ---");

        ExcelReader reader = new ExcelReader(filePath);

        // Read all test cases
        List<Map<String, String>> testCases = reader.readSheetAsMap("TestData");

        System.out.println("Running " + testCases.size() + " test cases...\n");

        // Simulate test execution
        for (Map<String, String> testCase : testCases) {
            String testCaseId = testCase.get("TestCaseID");
            String testCaseName = testCase.get("TestCaseName");
            String runMode = testCase.get("RunMode");

            if ("Yes".equalsIgnoreCase(runMode) || "Y".equalsIgnoreCase(runMode)) {
                System.out.println("Executing: " + testCaseId + " - " + testCaseName);

                // Access test data
                String username = testCase.get("Username");
                String password = testCase.get("Password");

                System.out.println("  Using credentials: " + username + " / " + password);
                System.out.println("  Test executed successfully\n");
            } else {
                System.out.println("Skipping: " + testCaseId + " - " + testCaseName + "\n");
            }
        }

        reader.close();
        System.out.println();
    }

    /**
     * Example: TestNG DataProvider Integration
     */
    public static Object[][] getTestDataForTestNG(String filePath, String sheetName) {
        ExcelReader reader = new ExcelReader(filePath);

        // Read data starting from row 1 (skip headers)
        String[][] data = reader.readSheetFromRow(sheetName, 1);

        reader.close();

        // Convert to Object[][] for TestNG
        Object[][] testData = new Object[data.length][];
        for (int i = 0; i < data.length; i++) {
            testData[i] = data[i];
        }

        return testData;
    }

    /**
     * Example: Write test execution results
     */
    public static void writeTestResults(String testCaseId, String status, String remarks) {
        String resultsFile = "test-output/TestResults.xlsx";
        ExcelWriter writer = new ExcelWriter(resultsFile);

        // Find the test case row
        int row = writer.findRowByColumnValue("TestResults", 0, testCaseId);

        if (row == -1) {
            // Test case not found, append new row
            String[] newRow = {testCaseId, "", status, remarks};
            writer.appendRow("TestResults", newRow);
        } else {
            // Update existing row
            writer.setCellData("TestResults", row, 2, status);
            writer.setCellData("TestResults", row, 3, remarks);
            writer.writeTestResult("TestResults", row, 2, status);
        }

        writer.close();
    }

    /**
     * Example: Cleanup and file operations
     */
    public static void cleanupExample(String filePath) {
        ExcelWriter writer = new ExcelWriter(filePath);

        // Clear existing data from sheet
        writer.clearSheet("TestResults");

        // Remove old sheets
        if (writer.isSheetExist("OldData")) {
            writer.removeSheet("OldData");
        }

        // Create fresh sheet
        writer.createSheet("NewResults");

        writer.close();
    }
}
