package com.automation.utils;

import org.apache.poi.ss.usermodel.*;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;

import java.io.*;
import java.util.*;

/**
 * Comprehensive Excel Utility for reading and writing Excel files
 * Supports both .xlsx and .xls formats
 *
 * @author DevTools
 * @version 1.0
 */
public class ExcelUtil {
    protected String filePath;
    protected FileInputStream fis;
    protected FileOutputStream fos;
    protected Workbook workbook;
    protected Sheet sheet;

    /**
     * Constructor - Opens Excel file
     * @param filePath Path to Excel file (.xlsx or .xls)
     */
    public ExcelUtil(String filePath) {
        this.filePath = filePath;
        try {
            File file = new File(filePath);
            if (!file.exists()) {
                throw new FileNotFoundException("Excel file not found: " + filePath);
            }

            fis = new FileInputStream(file);
            if (filePath.endsWith(".xlsx")) {
                workbook = new XSSFWorkbook(fis);
            } else if (filePath.endsWith(".xls")) {
                workbook = new HSSFWorkbook(fis);
            } else {
                throw new IllegalArgumentException("Unsupported file format. Use .xlsx or .xls");
            }
        } catch (Exception e) {
            System.err.println("Error opening Excel file: " + e.getMessage());
            e.printStackTrace();
        }
    }

    /**
     * Get total row count in sheet (including header)
     * @param sheetName Name of the sheet
     * @return Number of rows
     */
    public int getRowCount(String sheetName) {
        try {
            sheet = workbook.getSheet(sheetName);
            if (sheet == null) {
                System.err.println("Sheet not found: " + sheetName);
                return 0;
            }
            return sheet.getLastRowNum() + 1;
        } catch (Exception e) {
            System.err.println("Error getting row count: " + e.getMessage());
            return 0;
        }
    }

    /**
     * Get total column count in sheet
     * @param sheetName Name of the sheet
     * @return Number of columns
     */
    public int getColumnCount(String sheetName) {
        try {
            sheet = workbook.getSheet(sheetName);
            if (sheet == null) return 0;

            Row row = sheet.getRow(0);
            return row != null ? row.getLastCellNum() : 0;
        } catch (Exception e) {
            System.err.println("Error getting column count: " + e.getMessage());
            return 0;
        }
    }

    /**
     * Get cell data as String
     * @param sheetName Name of the sheet
     * @param rowNum Row number (0-based)
     * @param colNum Column number (0-based)
     * @return Cell value as String
     */
    public String getCellData(String sheetName, int rowNum, int colNum) {
        try {
            sheet = workbook.getSheet(sheetName);
            if (sheet == null) return "";

            Row row = sheet.getRow(rowNum);
            if (row == null) return "";

            Cell cell = row.getCell(colNum);
            return getCellValue(cell);
        } catch (Exception e) {
            System.err.println("Error reading cell data: " + e.getMessage());
            return "";
        }
    }

    /**
     * Set cell data
     * @param sheetName Name of the sheet
     * @param rowNum Row number (0-based)
     * @param colNum Column number (0-based)
     * @param data Data to write
     * @return true if successful
     */
    public boolean setCellData(String sheetName, int rowNum, int colNum, String data) {
        try {
            sheet = workbook.getSheet(sheetName);
            if (sheet == null) {
                System.err.println("Sheet not found: " + sheetName);
                return false;
            }

            Row row = sheet.getRow(rowNum);
            if (row == null) {
                row = sheet.createRow(rowNum);
            }

            Cell cell = row.getCell(colNum);
            if (cell == null) {
                cell = row.createCell(colNum);
            }

            cell.setCellValue(data);
            saveWorkbook();
            return true;
        } catch (Exception e) {
            System.err.println("Error setting cell data: " + e.getMessage());
            e.printStackTrace();
            return false;
        }
    }

    /**
     * Check if sheet exists
     * @param sheetName Name of the sheet
     * @return true if sheet exists
     */
    public boolean isSheetExist(String sheetName) {
        return workbook.getSheet(sheetName) != null;
    }

    /**
     * Get all sheet names
     * @return List of sheet names
     */
    public List<String> getAllSheetNames() {
        List<String> sheetNames = new ArrayList<>();
        int numberOfSheets = workbook.getNumberOfSheets();
        for (int i = 0; i < numberOfSheets; i++) {
            sheetNames.add(workbook.getSheetName(i));
        }
        return sheetNames;
    }

    /**
     * Find row number by searching for a value in specific column
     * @param sheetName Name of the sheet
     * @param colNum Column number to search in
     * @param value Value to search for
     * @return Row number if found, -1 otherwise
     */
    public int findRowByColumnValue(String sheetName, int colNum, String value) {
        int rows = getRowCount(sheetName);
        for (int i = 0; i < rows; i++) {
            String cellValue = getCellData(sheetName, i, colNum);
            if (cellValue.equals(value)) {
                return i;
            }
        }
        return -1;
    }

    /**
     * Convert cell value to String based on cell type
     * @param cell Excel cell
     * @return Cell value as String
     */
    protected String getCellValue(Cell cell) {
        if (cell == null) return "";

        switch (cell.getCellType()) {
            case STRING:
                return cell.getStringCellValue();

            case NUMERIC:
                if (DateUtil.isCellDateFormatted(cell)) {
                    return cell.getDateCellValue().toString();
                }
                // Return as long to avoid decimal points for whole numbers
                double numValue = cell.getNumericCellValue();
                if (numValue == (long) numValue) {
                    return String.valueOf((long) numValue);
                }
                return String.valueOf(numValue);

            case BOOLEAN:
                return String.valueOf(cell.getBooleanCellValue());

            case FORMULA:
                try {
                    return String.valueOf(cell.getNumericCellValue());
                } catch (Exception e) {
                    return cell.getStringCellValue();
                }

            case BLANK:
                return "";

            default:
                return "";
        }
    }

    /**
     * Save workbook to file
     */
    protected void saveWorkbook() {
        try {
            fos = new FileOutputStream(filePath);
            workbook.write(fos);
            fos.close();
        } catch (IOException e) {
            System.err.println("Error saving workbook: " + e.getMessage());
            e.printStackTrace();
        }
    }

    /**
     * Close workbook and streams
     */
    public void close() {
        try {
            if (workbook != null) {
                workbook.close();
            }
            if (fis != null) {
                fis.close();
            }
            if (fos != null) {
                fos.close();
            }
        } catch (IOException e) {
            System.err.println("Error closing workbook: " + e.getMessage());
            e.printStackTrace();
        }
    }

    /**
     * Get workbook instance
     * @return Workbook object
     */
    public Workbook getWorkbook() {
        return workbook;
    }
}
