package com.automation.utils;

import org.apache.poi.ss.usermodel.*;

import java.util.*;

/**
 * Advanced Excel Reading Operations
 * Extends ExcelUtil with methods for reading entire sheets and test data
 *
 * @author DevTools
 * @version 1.0
 */
public class ExcelReader extends ExcelUtil {

    /**
     * Constructor - Opens Excel file
     * @param filePath Path to Excel file (.xlsx or .xls)
     */
    public ExcelReader(String filePath) {
        super(filePath);
    }

    /**
     * Read entire sheet as 2D String array
     * @param sheetName Name of the sheet
     * @return 2D array of cell values
     */
    public String[][] readSheet(String sheetName) {
        try {
            sheet = workbook.getSheet(sheetName);
            if (sheet == null) {
                System.err.println("Sheet not found: " + sheetName);
                return new String[0][0];
            }

            int rowCount = getRowCount(sheetName);
            int colCount = getColumnCount(sheetName);

            String[][] data = new String[rowCount][colCount];

            for (int i = 0; i < rowCount; i++) {
                for (int j = 0; j < colCount; j++) {
                    data[i][j] = getCellData(sheetName, i, j);
                }
            }

            return data;
        } catch (Exception e) {
            System.err.println("Error reading sheet: " + e.getMessage());
            e.printStackTrace();
            return new String[0][0];
        }
    }

    /**
     * Read sheet as List of Maps (column headers as keys)
     * First row is treated as header
     * @param sheetName Name of the sheet
     * @return List of maps where each map represents a row
     */
    public List<Map<String, String>> readSheetAsMap(String sheetName) {
        List<Map<String, String>> dataList = new ArrayList<>();

        try {
            sheet = workbook.getSheet(sheetName);
            if (sheet == null) {
                System.err.println("Sheet not found: " + sheetName);
                return dataList;
            }

            int rowCount = getRowCount(sheetName);
            int colCount = getColumnCount(sheetName);

            if (rowCount == 0) return dataList;

            // Get headers from first row
            String[] headers = new String[colCount];
            for (int j = 0; j < colCount; j++) {
                headers[j] = getCellData(sheetName, 0, j);
            }

            // Read data rows (starting from row 1)
            for (int i = 1; i < rowCount; i++) {
                Map<String, String> rowData = new LinkedHashMap<>();
                for (int j = 0; j < colCount; j++) {
                    String value = getCellData(sheetName, i, j);
                    rowData.put(headers[j], value);
                }
                dataList.add(rowData);
            }

        } catch (Exception e) {
            System.err.println("Error reading sheet as map: " + e.getMessage());
            e.printStackTrace();
        }

        return dataList;
    }

    /**
     * Get specific test data by test case name
     * Searches in first column for test case name
     * @param sheetName Name of the sheet
     * @param testCaseName Test case name to search for
     * @return Map of column headers to values for that row
     */
    public Map<String, String> getTestData(String sheetName, String testCaseName) {
        Map<String, String> testData = new LinkedHashMap<>();

        try {
            int rowNum = findRowByColumnValue(sheetName, 0, testCaseName);
            if (rowNum == -1) {
                System.err.println("Test case not found: " + testCaseName);
                return testData;
            }

            int colCount = getColumnCount(sheetName);

            // Get headers from first row
            for (int j = 0; j < colCount; j++) {
                String header = getCellData(sheetName, 0, j);
                String value = getCellData(sheetName, rowNum, j);
                testData.put(header, value);
            }

        } catch (Exception e) {
            System.err.println("Error getting test data: " + e.getMessage());
            e.printStackTrace();
        }

        return testData;
    }

    /**
     * Read specific row as array
     * @param sheetName Name of the sheet
     * @param rowNum Row number (0-based)
     * @return Array of cell values in the row
     */
    public String[] readRow(String sheetName, int rowNum) {
        try {
            int colCount = getColumnCount(sheetName);
            String[] rowData = new String[colCount];

            for (int j = 0; j < colCount; j++) {
                rowData[j] = getCellData(sheetName, rowNum, j);
            }

            return rowData;
        } catch (Exception e) {
            System.err.println("Error reading row: " + e.getMessage());
            return new String[0];
        }
    }

    /**
     * Read specific column as array
     * @param sheetName Name of the sheet
     * @param colNum Column number (0-based)
     * @return Array of cell values in the column
     */
    public String[] readColumn(String sheetName, int colNum) {
        try {
            int rowCount = getRowCount(sheetName);
            String[] colData = new String[rowCount];

            for (int i = 0; i < rowCount; i++) {
                colData[i] = getCellData(sheetName, i, colNum);
            }

            return colData;
        } catch (Exception e) {
            System.err.println("Error reading column: " + e.getMessage());
            return new String[0];
        }
    }

    /**
     * Get column index by header name
     * Searches in first row for column header
     * @param sheetName Name of the sheet
     * @param headerName Column header name
     * @return Column index if found, -1 otherwise
     */
    public int getColumnIndex(String sheetName, String headerName) {
        try {
            int colCount = getColumnCount(sheetName);
            for (int j = 0; j < colCount; j++) {
                String header = getCellData(sheetName, 0, j);
                if (header.equals(headerName)) {
                    return j;
                }
            }
        } catch (Exception e) {
            System.err.println("Error finding column: " + e.getMessage());
        }
        return -1;
    }

    /**
     * Get cell data by header name (useful for test data)
     * @param sheetName Name of the sheet
     * @param rowNum Row number (0-based)
     * @param headerName Column header name
     * @return Cell value
     */
    public String getCellDataByHeader(String sheetName, int rowNum, String headerName) {
        int colNum = getColumnIndex(sheetName, headerName);
        if (colNum == -1) {
            System.err.println("Header not found: " + headerName);
            return "";
        }
        return getCellData(sheetName, rowNum, colNum);
    }

    /**
     * Find all rows where a column contains specific value
     * @param sheetName Name of the sheet
     * @param colNum Column number to search in
     * @param value Value to search for
     * @return List of row numbers containing the value
     */
    public List<Integer> findAllRowsByColumnValue(String sheetName, int colNum, String value) {
        List<Integer> rowNumbers = new ArrayList<>();
        int rowCount = getRowCount(sheetName);

        for (int i = 0; i < rowCount; i++) {
            String cellValue = getCellData(sheetName, i, colNum);
            if (cellValue.equals(value)) {
                rowNumbers.add(i);
            }
        }

        return rowNumbers;
    }

    /**
     * Check if a cell is empty
     * @param sheetName Name of the sheet
     * @param rowNum Row number (0-based)
     * @param colNum Column number (0-based)
     * @return true if cell is empty or blank
     */
    public boolean isCellEmpty(String sheetName, int rowNum, int colNum) {
        String value = getCellData(sheetName, rowNum, colNum);
        return value == null || value.trim().isEmpty();
    }

    /**
     * Get data starting from specific row (useful to skip headers)
     * @param sheetName Name of the sheet
     * @param startRow Starting row number (0-based)
     * @return 2D array of cell values from startRow to end
     */
    public String[][] readSheetFromRow(String sheetName, int startRow) {
        try {
            int totalRows = getRowCount(sheetName);
            int colCount = getColumnCount(sheetName);
            int dataRows = totalRows - startRow;

            if (dataRows <= 0) {
                return new String[0][0];
            }

            String[][] data = new String[dataRows][colCount];

            for (int i = 0; i < dataRows; i++) {
                for (int j = 0; j < colCount; j++) {
                    data[i][j] = getCellData(sheetName, startRow + i, j);
                }
            }

            return data;
        } catch (Exception e) {
            System.err.println("Error reading sheet from row: " + e.getMessage());
            return new String[0][0];
        }
    }
}
