# GitHub Webhook PHP Package

Native PHP webhook handler with HMAC signature verification for Apache and Nginx servers. Enables automated deployments triggered by GitHub repository events.

## Files Included

| File | Purpose |
|------|---------|
| `webhook.php` | Main webhook handler with signature verification and deployment |
| `verify-signature.php` | HMAC SHA-256 signature verification utility function |
| `deploy.sh` | Automated deployment script for git operations |
| `.htaccess` | Apache configuration for webhook endpoint |
| `nginx-webhook.conf` | Nginx server block configuration sample |
| `README.md` | This setup guide |

## Features

- Native PHP implementation (no dependencies)
- HMAC SHA-256 signature verification
- Apache & Nginx support
- JSON payload parsing
- Shell script execution
- Logging and error handling
- Security best practices

## Quick Start

### 1. Installation

```bash
# Create webhook directory
mkdir -p /var/www/webhook

# Copy files
cp webhook.php /var/www/webhook/
cp deploy.sh /var/www/webhook/
chmod +x /var/www/webhook/deploy.sh

# Create log file
touch /var/log/github-webhook.log
chown www-data:www-data /var/log/github-webhook.log
```

### 2. Configuration

#### Generate Webhook Secret

```bash
# Generate a strong random secret
openssl rand -hex 32
```

#### Configure webhook.php

Edit `webhook.php` and update:

```php
$secret = getenv('WEBHOOK_SECRET') ?: 'your_generated_secret';
$deploy_script = '/var/www/webhook/deploy.sh';
$log_file = '/var/log/github-webhook.log';
```

#### Configure deploy.sh

Edit `deploy.sh` and update:

```bash
REPO_DIR="/var/www/html"  # Your repository path
BRANCH="main"              # Target branch
LOG_FILE="/var/log/deploy.log"
```

### 3. Web Server Setup

#### Apache

1. Copy `.htaccess` to webhook directory:
   ```bash
   cp .htaccess /var/www/webhook/
   ```

2. Edit `.htaccess` and set your secret:
   ```apache
   SetEnv WEBHOOK_SECRET your_generated_secret
   ```

3. Configure virtual host:
   ```bash
   sudo nano /etc/apache2/sites-available/webhook.conf
   ```

   ```apache
   <VirtualHost *:80>
       ServerName webhook.example.com
       DocumentRoot /var/www/webhook

       <Directory /var/www/webhook>
           AllowOverride All
           Require all granted
       </Directory>

       ErrorLog ${APACHE_LOG_DIR}/webhook_error.log
       CustomLog ${APACHE_LOG_DIR}/webhook_access.log combined
   </VirtualHost>
   ```

4. Enable site and restart:
   ```bash
   sudo a2ensite webhook
   sudo systemctl restart apache2
   ```

#### Nginx + PHP-FPM

1. Copy configuration:
   ```bash
   sudo cp nginx-webhook.conf /etc/nginx/sites-available/webhook
   ```

2. Edit configuration and update:
   ```nginx
   server_name webhook.example.com;
   root /var/www/webhook;
   fastcgi_param WEBHOOK_SECRET your_generated_secret;
   ```

3. Enable site and restart:
   ```bash
   sudo ln -s /etc/nginx/sites-available/webhook /etc/nginx/sites-enabled/
   sudo nginx -t
   sudo systemctl restart nginx
   sudo systemctl restart php8.1-fpm
   ```

### 4. GitHub Configuration

1. Go to your repository on GitHub
2. Navigate to **Settings** > **Webhooks** > **Add webhook**
3. Configure:
   - **Payload URL**: `http://webhook.example.com/webhook.php`
   - **Content type**: `application/json`
   - **Secret**: Your generated secret
   - **Events**: Select "Just the push event"
   - **Active**: Check this box
4. Click **Add webhook**

### 5. Permissions Setup

```bash
# Make deploy script executable
chmod +x /var/www/webhook/deploy.sh

# Give web server user access to repository
sudo chown -R www-data:www-data /var/www/html

# Or use sudo for git operations (add to /etc/sudoers)
www-data ALL=(ALL) NOPASSWD: /usr/bin/git
```

## Testing

### Test Webhook Locally

```bash
# Test signature verification
curl -X POST http://localhost/webhook.php \
  -H "X-GitHub-Event: push" \
  -H "X-Hub-Signature-256: sha256=$(echo -n '{"ref":"refs/heads/main"}' | openssl dgst -sha256 -hmac 'your_secret' | cut -d' ' -f2)" \
  -d '{"ref":"refs/heads/main"}'
```

### Test from GitHub

1. Go to webhook settings
2. Click on your webhook
3. Scroll to **Recent Deliveries**
4. Click **Redeliver** on any delivery

### View Logs

```bash
# Webhook logs
tail -f /var/log/github-webhook.log

# Deployment logs
tail -f /var/log/deploy.log

# Apache logs
tail -f /var/log/apache2/webhook_error.log

# Nginx logs
tail -f /var/log/nginx/webhook_error.log
```

## Security Best Practices

1. **Use HTTPS**: Always use SSL/TLS in production
   ```bash
   sudo certbot --nginx -d webhook.example.com
   ```

2. **Strong Secret**: Use a long random secret (32+ characters)

3. **File Permissions**: Restrict access to webhook files
   ```bash
   chmod 600 /var/www/webhook/.htaccess
   chmod 600 /var/www/webhook/nginx-webhook.conf
   ```

4. **Firewall**: Only allow GitHub IPs (optional)
   ```bash
   # Get GitHub webhook IPs
   curl https://api.github.com/meta | jq -r '.hooks[]'
   ```

5. **Limit Events**: Only enable webhook events you need

6. **Logging**: Monitor logs regularly for suspicious activity

## Troubleshooting

### Common Issues

#### 1. "Invalid signature" error

- Verify secret matches in GitHub and PHP
- Check webhook payload is being received correctly
- Ensure no proxy is modifying the request

#### 2. Deployment script not executing

- Check file permissions: `ls -la /var/www/webhook/deploy.sh`
- Verify script path in `webhook.php`
- Check logs: `tail -f /var/log/github-webhook.log`

#### 3. Git pull fails

- Verify web server user has access: `sudo -u www-data git pull`
- Check SSH keys or credentials
- Ensure repository directory is writable

#### 4. 403 Forbidden

- Check directory permissions: `ls -la /var/www/webhook`
- Verify `.htaccess` or nginx config
- Check Apache/Nginx error logs

#### 5. 502 Bad Gateway (Nginx)

- Verify PHP-FPM is running: `systemctl status php8.1-fpm`
- Check PHP-FPM socket path in nginx config
- Review PHP-FPM error logs: `tail -f /var/log/php8.1-fpm.log`

### Debug Mode

Enable verbose logging in `webhook.php`:

```php
// Add at top of file
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Add detailed logging
file_put_contents($log_file, "Headers: " . print_r(getallheaders(), true), FILE_APPEND);
file_put_contents($log_file, "Payload: $payload\n", FILE_APPEND);
```

## Advanced Usage

### Multiple Repositories

Handle multiple repositories in `webhook.php`:

```php
if ($event === "push") {
    $data = json_decode($payload, true);
    $repo = $data['repository']['name'] ?? '';

    switch($repo) {
        case 'frontend':
            shell_exec("sh /path/to/deploy-frontend.sh 2>&1");
            break;
        case 'backend':
            shell_exec("sh /path/to/deploy-backend.sh 2>&1");
            break;
    }
}
```

### Branch-Specific Deployment

Deploy only specific branches:

```php
if ($event === "push") {
    $data = json_decode($payload, true);
    $branch = $data['ref'] ?? '';

    if ($branch === 'refs/heads/main') {
        shell_exec("sh /path/to/deploy-production.sh 2>&1");
    } elseif ($branch === 'refs/heads/staging') {
        shell_exec("sh /path/to/deploy-staging.sh 2>&1");
    }
}
```

### Slack Notifications

Add notifications to `deploy.sh`:

```bash
# Send Slack notification
curl -X POST -H 'Content-type: application/json' \
  --data '{"text":"Deployment completed for repository X"}' \
  YOUR_SLACK_WEBHOOK_URL
```

## Resources

- [GitHub Webhooks Documentation](https://docs.github.com/en/webhooks)
- [Webhook Events Reference](https://docs.github.com/en/webhooks/webhook-events-and-payloads)
- [PHP Documentation](https://www.php.net/)

## Support

For issues or questions:
- Check GitHub webhook delivery logs
- Review server error logs
- Verify all configuration steps
- Test with GitHub webhook "Redeliver" feature

## License

This package is provided as-is for educational and production use.
