import crypto from "crypto";

/**
 * Verifies GitHub webhook signature using HMAC SHA-256
 * @param {string} signature - The x-hub-signature-256 header value
 * @param {string|Buffer} payload - The raw request body
 * @param {string} secret - Your webhook secret
 * @returns {boolean} True if signature is valid
 */
export function verifyGitHubSignature(signature, payload, secret) {
  if (!signature) return false;

  const hmac = crypto.createHmac("sha256", secret);
  const digest = "sha256=" + hmac.update(payload).digest("hex");

  return signature === digest;
}

/**
 * Express middleware for verifying GitHub webhook signatures
 * @param {string} secret - Your webhook secret
 * @returns {Function} Express middleware function
 */
export function createVerificationMiddleware(secret) {
  return (req, res, next) => {
    const signature = req.headers["x-hub-signature-256"];

    if (!verifyGitHubSignature(signature, req.body, secret)) {
      return res.status(401).send("Invalid signature");
    }

    next();
  };
}
