# GitHub Webhook Node.js Handler

Express and vanilla Node.js webhook handlers with HMAC signature verification and PM2 deployment scripts.

## Features

- **Express & Vanilla HTTP** - Choose between Express framework or vanilla Node.js HTTP server
- **HMAC SHA-256 Verification** - Secure signature verification for all webhook requests
- **PM2 Process Management** - Production-ready process management with auto-restart
- **Automated Deployment** - Trigger git pull and application restart on push events
- **Production-Ready** - Error handling, logging, and environment variable support

## Quick Start

### 1. Installation

```bash
# Install dependencies
npm install

# Install PM2 globally (for production)
npm install -g pm2
```

### 2. Configuration

Create a `.env` file in the project root:

```env
SECRET=your_github_webhook_secret
PORT=3000
```

Update `deploy.sh` with your project paths:

```bash
PROJECT_DIR="/path/to/your/project"
BRANCH="main"
PM2_APP_NAME="github-webhook"
```

Update `ecosystem.config.js` with your webhook secret:

```js
env: {
  SECRET: "your_webhook_secret_here"
}
```

### 3. Run the Server

**Development (Express):**
```bash
npm start
# or
npm run dev  # with auto-reload
```

**Development (Vanilla):**
```bash
npm run start:vanilla
```

**Production (PM2):**
```bash
npm run pm2:start
```

## File Overview

| File | Purpose |
|------|---------|
| `express-webhook.js` | Express-based webhook handler with signature verification |
| `vanilla-webhook.js` | Vanilla Node.js HTTP server webhook handler |
| `webhook-verifier.js` | Shared HMAC SHA-256 signature verification utility |
| `deploy.sh` | Automated deployment script for git pull and restart |
| `package.json` | NPM dependencies and scripts configuration |
| `ecosystem.config.js` | PM2 process manager configuration |

## Usage

### Express Handler

The Express implementation (`express-webhook.js`) provides a clean, middleware-based approach:

```js
import crypto from "crypto";
import express from "express";

const app = express();
app.use(express.raw({ type: "application/json" }));

app.post("/webhook", (req, res) => {
  if (!verifySignature(req)) {
    return res.status(401).send("Invalid");
  }

  const event = req.headers["x-github-event"];
  if (event === "push") {
    require("child_process").exec("sh /path/to/deploy.sh");
  }

  res.send("OK");
});
```

### Vanilla Handler

The vanilla implementation (`vanilla-webhook.js`) uses only Node.js built-in modules:

```js
import crypto from "crypto";
import http from "http";

http.createServer((req, res) => {
  // Handle webhook requests without Express
}).listen(3001);
```

### Signature Verification

Both implementations verify GitHub's HMAC SHA-256 signature:

```js
function verifySignature(signature, payload) {
  const hmac = crypto.createHmac("sha256", SECRET);
  const digest = "sha256=" + hmac.update(payload).digest("hex");
  return signature === digest;
}
```

## GitHub Webhook Setup

1. Go to your GitHub repository → Settings → Webhooks → Add webhook
2. Set **Payload URL**: `http://your-server.com:3000/webhook`
3. Set **Content type**: `application/json`
4. Set **Secret**: Your webhook secret (same as in `.env`)
5. Select events: Choose "Just the push event" or customize
6. Click **Add webhook**

## PM2 Commands

```bash
# Start application
npm run pm2:start
# or
pm2 start ecosystem.config.js

# Stop application
npm run pm2:stop

# Restart application
npm run pm2:restart

# View logs
npm run pm2:logs
# or
pm2 logs github-webhook

# Monitor application
pm2 monit

# Auto-restart on server reboot
pm2 startup
pm2 save
```

## Deployment Script

The `deploy.sh` script is executed when a push event is received:

1. Changes to project directory
2. Fetches latest changes from GitHub
3. Resets to latest commit on branch
4. Installs/updates dependencies
5. Restarts PM2 application

**Make the script executable:**
```bash
chmod +x deploy.sh
```

## Security Best Practices

1. **Always verify signatures** - Never skip HMAC verification in production
2. **Use environment variables** - Store secrets in `.env` files, not in code
3. **Restrict webhook events** - Only subscribe to events you need
4. **Use HTTPS** - Configure SSL/TLS for production deployments
5. **Firewall rules** - Restrict access to webhook endpoint if possible
6. **Log everything** - Monitor webhook activity and deployment logs

## Nginx Reverse Proxy

For production, use Nginx as a reverse proxy:

```nginx
server {
    listen 80;
    server_name webhook.example.com;

    location /webhook {
        proxy_pass http://localhost:3000;
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
    }
}
```

## Docker Deployment

Build and run with Docker:

```bash
# Build image
docker build -t github-webhook-node .

# Run container
docker run -p 3000:3000 \
  -e SECRET=your_secret \
  -e PORT=3000 \
  github-webhook-node
```

## Troubleshooting

**Webhook returns 401 Unauthorized:**
- Verify the secret in `.env` matches your GitHub webhook secret
- Check that signature verification is working correctly
- Review webhook delivery logs in GitHub

**Deployment script not executing:**
- Ensure `deploy.sh` has execute permissions (`chmod +x`)
- Check the path in the webhook handler is correct
- Review deployment logs in `/var/log/webhook-deploy.log`

**PM2 not restarting:**
- Verify PM2 is installed globally: `pm2 --version`
- Check PM2 application name matches: `pm2 list`
- Review PM2 logs: `pm2 logs github-webhook`

## Additional Resources

- [GitHub Webhooks Documentation](https://docs.github.com/en/webhooks)
- [Webhook Events and Payloads](https://docs.github.com/en/webhooks/webhook-events-and-payloads)
- [Express.js Documentation](https://expressjs.com/)
- [PM2 Documentation](https://pm2.keymetrics.io/)

## License

MIT
