# GitHub Webhook Java Implementation

A production-ready Spring Boot implementation for handling GitHub webhooks with HMAC SHA-256 signature verification and automated deployment capabilities.

## Features

- Spring Boot REST controller for webhook handling
- HMAC SHA-256 signature verification for security
- Automated deployment script execution
- Event-based processing (push, pull_request, etc.)
- Production-ready error handling
- Docker and systemd deployment options

## Prerequisites

- Java 11 or higher
- Maven 3.6 or higher
- Spring Boot 2.x or 3.x
- GitHub repository with webhook configuration

## Project Structure

```
github-webhook-java/
├── WebhookController.java      # Main Spring Boot REST controller
├── WebhookVerifier.java        # HMAC signature verification utility
├── deploy.sh                   # Automated deployment script
├── application.properties      # Spring Boot configuration
└── README.md                   # This file
```

## Quick Start

### 1. Maven Dependencies

Add the following dependencies to your `pom.xml`:

```xml
<dependencies>
    <!-- Spring Boot Web -->
    <dependency>
        <groupId>org.springframework.boot</groupId>
        <artifactId>spring-boot-starter-web</artifactId>
    </dependency>

    <!-- Apache Commons Codec for HMAC -->
    <dependency>
        <groupId>commons-codec</groupId>
        <artifactId>commons-codec</artifactId>
    </dependency>
</dependencies>
```

### 2. Configuration

Edit `application.properties`:

```properties
server.port=8080
webhook.secret=your_github_webhook_secret_here
```

**IMPORTANT:** Replace `your_github_webhook_secret_here` with your actual GitHub webhook secret.

### 3. Update Controller

In `WebhookController.java`, replace:
```java
private static final String SECRET = "YOUR_WEBHOOK_SECRET";
```

With your actual webhook secret, or better yet, load it from application.properties:
```java
@Value("${webhook.secret}")
private String secret;
```

### 4. Configure Deployment Script

Edit `deploy.sh` and update:
- `REPO_DIR`: Path to your repository
- `BRANCH`: Git branch to deploy
- Build and restart commands for your specific project

Make the script executable:
```bash
chmod +x deploy.sh
```

### 5. Build and Run

```bash
# Build the project
mvn clean package

# Run the application
java -jar target/your-app.jar
```

## GitHub Webhook Setup

1. Go to your GitHub repository
2. Navigate to **Settings** → **Webhooks** → **Add webhook**
3. Configure:
   - **Payload URL**: `http://your-server:8080/webhook`
   - **Content type**: `application/json`
   - **Secret**: Enter the same secret you configured in application.properties
   - **Events**: Select "Just the push event" or customize as needed
4. Click **Add webhook**

## Deployment Options

### Docker Deployment

Create a `Dockerfile`:

```dockerfile
FROM openjdk:17-slim
COPY target/webhook-handler.jar app.jar
EXPOSE 8080
ENTRYPOINT ["java", "-jar", "/app.jar"]
```

Build and run:
```bash
docker build -t github-webhook .
docker run -p 8080:8080 -e SECRET=your_secret github-webhook
```

### Systemd Service

Create `/etc/systemd/system/github-webhook.service`:

```ini
[Unit]
Description=GitHub Webhook Handler
After=network.target

[Service]
Type=simple
User=appuser
WorkingDirectory=/opt/webhook
ExecStart=/usr/bin/java -jar /opt/webhook/app.jar
Restart=on-failure

[Install]
WantedBy=multi-user.target
```

Enable and start:
```bash
sudo systemctl enable github-webhook
sudo systemctl start github-webhook
sudo systemctl status github-webhook
```

## Security Best Practices

1. **Always verify signatures**: The implementation includes HMAC SHA-256 verification
2. **Use HTTPS**: Configure your server with SSL/TLS certificates
3. **Restrict IP ranges**: Configure firewall to only allow GitHub's webhook IPs
4. **Keep secrets secure**: Use environment variables or secret management systems
5. **Log webhook events**: Monitor for suspicious activity
6. **Rate limiting**: Implement rate limiting to prevent abuse

## Supported GitHub Events

The controller currently handles `push` events, but you can extend it to handle:

- `pull_request`
- `issues`
- `release`
- `deployment`
- And many more ([see GitHub docs](https://docs.github.com/en/webhooks/webhook-events-and-payloads))

### Example: Handle Multiple Events

```java
String event = request.getHeader("X-GitHub-Event");

switch (event) {
    case "push":
        runDeployScript();
        break;
    case "pull_request":
        runTests();
        break;
    case "release":
        notifyTeam();
        break;
    default:
        log.info("Received event: " + event);
}
```

## Troubleshooting

### Signature Verification Fails

- Ensure the secret in `application.properties` matches GitHub webhook secret
- Check that you're reading the raw request body (not parsed JSON)
- Verify the header name is `X-Hub-Signature-256` (not `X-Hub-Signature`)

### Deployment Script Not Running

- Check script permissions: `chmod +x deploy.sh`
- Verify the path in `WebhookController.java`
- Check application logs for errors

### Connection Timeout

- Ensure your server is accessible from GitHub's webhook IPs
- Check firewall rules and port forwarding
- Verify the application is running on the correct port

## Monitoring and Logging

View application logs:
```bash
# If using systemd
sudo journalctl -u github-webhook -f

# If using file logging
tail -f /var/log/webhook-handler.log

# Deployment logs
tail -f /var/log/webhook-deploy.log
```

## Testing

### Test Locally with curl

```bash
# Generate HMAC signature
SECRET="your_secret"
PAYLOAD='{"ref":"refs/heads/main"}'
SIGNATURE=$(echo -n "$PAYLOAD" | openssl dgst -sha256 -hmac "$SECRET" | sed 's/^.* //')

# Send test webhook
curl -X POST http://localhost:8080/webhook \
  -H "Content-Type: application/json" \
  -H "X-GitHub-Event: push" \
  -H "X-Hub-Signature-256: sha256=$SIGNATURE" \
  -d "$PAYLOAD"
```

### Redeliver from GitHub

1. Go to **Settings** → **Webhooks**
2. Click on your webhook
3. Scroll to **Recent Deliveries**
4. Click **Redeliver** on any delivery

## Resources

- [GitHub Webhooks Documentation](https://docs.github.com/en/webhooks)
- [Webhook Events and Payloads](https://docs.github.com/en/webhooks/webhook-events-and-payloads)
- [Spring Boot Documentation](https://spring.io/projects/spring-boot)
- [Securing Webhooks](https://docs.github.com/en/webhooks/using-webhooks/validating-webhook-deliveries)

## License

This implementation is provided as-is for educational and production use.

## Support

For issues or questions:
- Check GitHub webhook delivery status
- Review application logs
- Verify network connectivity
- Test signature verification separately
